// Lógica para mostrar nombre de usuario o nombre/desc de Wishio
// Mostrar avatar si existe, si no el SVG proporcionado, junto al nombre/enlace
function escapeHtml(text) {
    return text.replace(/[&<>"']/g, function (c) {
        return {'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c];
    });
}
function setHeaderUser(username) {
    var info = document.getElementById('header-info');
    var logo = document.querySelector('#header-container img');
    if (!info) return;
    // Limpiar contenido anterior
    info.innerHTML = '';
    if (username) {
        // Ocultar logo Wishio
        if (logo) logo.style.display = 'none';
        chrome.storage.local.get(['wishio_avatar'], (result) => {
            // Crear contenedor
            const container = document.createElement('span');
            container.style.display = 'flex';
            container.style.alignItems = 'center';
            container.style.gap = '0';

            // Avatar
            let avatarNode;
            if (result.wishio_avatar) {
                let avatarUrl = result.wishio_avatar;
                if (!/^https?:\/\//.test(avatarUrl)) {
                    if (!avatarUrl.startsWith('/images/avatars/')) {
                        avatarUrl = '/images/avatars/' + avatarUrl.replace(/^\/+/, '');
                    }
                    avatarUrl = 'https://wishio.es' + avatarUrl;
                }
                avatarNode = document.createElement('img');
                avatarNode.src = avatarUrl;
                avatarNode.alt = 'avatar';
                avatarNode.style.width = '36px';
                avatarNode.style.height = '36px';
                avatarNode.style.borderRadius = '50%';
                avatarNode.style.objectFit = 'cover';
                avatarNode.style.marginRight = '10px';
                avatarNode.style.verticalAlign = 'middle';
                avatarNode.style.background = '#f3f4f6';
                avatarNode.onerror = function() {
                    this.style.display = 'none';
                    const fallback = document.createElement('span');
                    fallback.style.display = 'inline-block';
                    fallback.style.width = '36px';
                    fallback.style.height = '36px';
                    fallback.style.marginRight = '10px';
                    fallback.style.verticalAlign = 'middle';
                    // Crear SVG con createElementNS
                    const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
                    svg.setAttribute('fill', 'none');
                    svg.setAttribute('viewBox', '0 0 24 24');
                    svg.setAttribute('stroke-width', '1.5');
                    svg.setAttribute('stroke', '#888');
                    svg.setAttribute('style', 'width:36px;height:36px;');
                    const path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
                    path.setAttribute('stroke-linecap', 'round');
                    path.setAttribute('stroke-linejoin', 'round');
                    path.setAttribute('d', 'M17.982 18.725A7.488 7.488 0 0 0 12 15.75a7.488 7.488 0 0 0-5.982 2.975m11.963 0a9 9 0 1 0-11.963 0m11.963 0A8.966 8.966 0 0 1 12 21a8.966 8.966 0 0 1-5.982-2.275M15 9.75a3 3 0 1 1-6 0 3 3 0 0 1 6 0Z');
                    svg.appendChild(path);
                    fallback.appendChild(svg);
                    this.parentNode.insertBefore(fallback, this.nextSibling);
                };
            } else {
                avatarNode = document.createElement('span');
                avatarNode.style.display = 'inline-block';
                avatarNode.style.width = '36px';
                avatarNode.style.height = '36px';
                avatarNode.style.marginRight = '10px';
                avatarNode.style.verticalAlign = 'middle';
                avatarNode.innerHTML = `<svg xmlns=\"http://www.w3.org/2000/svg\" fill=\"none\" viewBox=\"0 0 24 24\" stroke-width=\"1.5\" stroke=\"#888\" style=\"width:36px;height:36px;\"><path stroke-linecap=\"round\" stroke-linejoin=\"round\" d=\"M17.982 18.725A7.488 7.488 0 0 0 12 15.75a7.488 7.488 0 0 0-5.982 2.975m11.963 0a9 9 0 1 0-11.963 0m11.963 0A8.966 8.966 0 0 1 12 21a8.966 8.966 0 0 1-5.982-2.275M15 9.75a3 3 0 1 1-6 0 3 3 0 0 1 6 0Z\"/></svg>`;
            }
            container.appendChild(avatarNode);

            // Enlace username
            const safeUsername = typeof escapeHtml === 'function' ? escapeHtml(username) : username;
            const userLink = document.createElement('a');
            userLink.href = `https://wishio.es/${encodeURIComponent(username)}?sort=created_at&direction=desc`;
            userLink.target = '_blank';
            userLink.className = 'wishio-title';
            userLink.style.color = '#6366f1';
            userLink.style.textDecoration = 'underline';
            userLink.style.fontWeight = 'bold';
            userLink.style.cursor = 'pointer';
            userLink.style.fontSize = '1.5em';
            userLink.style.marginBottom = '2px';
            userLink.style.letterSpacing = '0.5px';
            userLink.textContent = safeUsername;
            container.appendChild(userLink);

            info.appendChild(container);
        });
    } else {
        // Mostrar logo Wishio
        if (logo) logo.style.display = '';
    // Mostrar nombre y descripción de Wishio sin innerHTML
    const wishioLink = document.createElement('a');
    wishioLink.href = 'https://wishio.es';
    wishioLink.target = '_blank';
    wishioLink.className = 'wishio-title';
    wishioLink.style.color = '#6366f1';
    wishioLink.style.textDecoration = 'underline';
    wishioLink.style.fontWeight = 'bold';
    wishioLink.style.cursor = 'pointer';
    wishioLink.style.fontSize = '1.5em';
    wishioLink.style.marginBottom = '2px';
    wishioLink.style.letterSpacing = '0.5px';
    wishioLink.textContent = 'Wishio';
    info.appendChild(wishioLink);
    const slogan = document.createElement('div');
    slogan.className = 'wishio-slogan';
    slogan.style.fontSize = '0.98rem';
    slogan.style.color = '#64748b';
    slogan.style.marginBottom = '0';
    // Añadir texto y salto de línea sin innerHTML
    slogan.appendChild(document.createTextNode('Todos tus deseos'));
    slogan.appendChild(document.createElement('br'));
    slogan.appendChild(document.createTextNode('en un solo lugar'));
    info.appendChild(slogan);
    }
}
// popup.js para la extensión Wishio

// Detectar entorno automáticamente
function getApiUrl() {
    // Siempre usar localhost en desarrollo
    if (true || location.hostname === 'localhost' || location.hostname === '127.0.0.1') {
        //return 'http://localhost:8000/api';
    }

    return 'https://wishio.es/api';
}
const API_URL = getApiUrl();

function showLogin() {
    document.getElementById('login-section').style.display = '';
    document.getElementById('add-section').style.display = 'none';
}
function showAdd() {
    document.getElementById('login-section').style.display = 'none';
    document.getElementById('add-section').style.display = '';
}

function saveToken(token, username) {
    chrome.storage.local.set({wishio_token: token, wishio_username: username});
}
function clearToken() {
    chrome.storage.local.remove(['wishio_token', 'wishio_username']);
}
function getToken(cb) {
    chrome.storage.local.get(['wishio_token'], (result) => {
        cb(result.wishio_token);
    });
}

// Login con token para API/Extensión
function login(username, password) {
    // Permitir login con usuario o email
    let loginData = {};
    if (username.includes('@')) {
        loginData = { email: username, password };
    } else {
        loginData = { username, password };
    }
    const progress = document.getElementById('login-progress');
    progress.style.display = 'block';
    document.getElementById('login-error').textContent = '';
    fetch(API_URL + '/login', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'Accept': 'application/json' },
        body: JSON.stringify(loginData)
    })
    .then(r => r.json().then(data => ({status: r.status, data})))
    .then(({status, data}) => {
        progress.style.display = 'none';
        if (status === 200 && data.token) {
            saveToken(data.token, data.user.name);
            if (data.user.avatar) {
                chrome.storage.local.set({wishio_avatar: data.user.avatar});
            } else {
                chrome.storage.local.remove(['wishio_avatar']);
            }
            showAdd();
            document.getElementById('login-error').textContent = '';
            fillUrlFromTab();
            if (typeof setHeaderUser === 'function') setHeaderUser(data.user.name);
            // Mostrar el botón de logout inmediatamente tras login
            const logoutBtn = document.getElementById('logout-btn');
            if (logoutBtn) logoutBtn.style.display = 'flex';
        } else {
            document.getElementById('login-error').textContent = data.message || 'Error de autenticación';
        }
    })
    .catch(() => {
        progress.style.display = 'none';
        document.getElementById('login-error').textContent = 'Error de red';
    });
}
// --- Comprobar si la URL (sin parámetros) ya está en la lista del usuario ---
function stripUrlParams(url) {
    try {
        const u = new URL(url);
        return u.origin + u.pathname;
    } catch {
        return url;
    }
}

function checkProductExists(token, url, cb) {
    // Llama al endpoint backend eficiente para comprobar existencia por URL (sin parámetros)
    const cleanUrl = stripUrlParams(url);
    fetch(API_URL + '/wishlist/exists', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': 'Bearer ' + token
        },
        body: JSON.stringify({ url: cleanUrl })
    })
    .then(r => r.json())
    .then(data => {
        cb(!!data.exists);
    })
    .catch(() => cb(false));
}

// Obtener URL activa y rellenar nombre y url
function fillUrlFromTab() {
    chrome.tabs.query({active: true, currentWindow: true}, function(tabs) {
        if (tabs[0]) {
            document.getElementById('product-url').value = tabs[0].url;
            // Compatibilidad multiplataforma para autocompletar el nombre del producto
            if (chrome.scripting && chrome.scripting.executeScript) {
                // Chrome, Edge, Brave, Opera, Vivaldi
                chrome.scripting.executeScript({
                    target: {tabId: tabs[0].id},
                    func: () => document.title.substring(0, 40)
                }, (results) => {
                    if (results && results[0] && results[0].result) {
                        document.getElementById('product-name').value = results[0].result;
                    }
                });
            } else if (chrome.tabs.executeScript) {
                // Firefox, Safari
                chrome.tabs.executeScript(tabs[0].id, {
                    code: 'document.title.substring(0, 40);'
                }, (results) => {
                    if (results && results[0]) {
                        document.getElementById('product-name').value = results[0];
                    }
                });
            } else {
                // Fallback: no se puede autocompletar
                document.getElementById('product-name').value = '';
            }
        }
    });
}

function getStoreFromUrl(url) {
    try {
        const u = new URL(url);
        let host = u.hostname.replace(/^www\./, '');
        const parts = host.split('.');
        if (parts.length >= 2) {
            // Para dominios tipo amazon.co.uk, tomamos los últimos 3 si el penúltimo es de 2-3 letras
            if (parts.length > 2 && parts[parts.length-2].length <= 3) {
                return parts[parts.length-3];
            }
            // Normal: amazon.com, ebay.es
            return parts[parts.length-2];
        }
        return host;
    } catch {
        return 'unknown';
    }
}

// Añadir producto
function addProduct(token, name, price, url) {
    const progress = document.getElementById('login-progress');
    progress.style.display = 'block';
    document.getElementById('add-error').textContent = '';
    document.getElementById('add-success').textContent = '';
    if (!url) {
        progress.style.display = 'none';
        document.getElementById('add-error').textContent = 'La URL es obligatoria.';
        return;
    }
    // Comprobar existencia antes de añadir
    checkProductExists(token, url, function(exists) {
        if (exists) {
            progress.style.display = 'none';
            document.getElementById('add-error').textContent = 'Este producto ya está en tu lista.';
            document.getElementById('add-success').textContent = '';
            return;
        }
        // Si el nombre está vacío, ponerlo al nombre principal de la tienda
        if (!name || name.trim() === '') {
            name = getStoreFromUrl(url);
        } else {
            name = name.trim().substring(0, 40);
        }
        // Si el precio está vacío, ponerlo a 0
        if (!price || price.trim() === '') {
            price = 0;
        } else {
            price = price.replace(',', '.');
            if (isNaN(price) || Number(price) < 0) {
                progress.style.display = 'none';
                document.getElementById('add-error').textContent = 'El precio debe ser un número positivo.';
                return;
            }
            price = Number(price);
        }
        fetch(API_URL + '/wishlist/add', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'Authorization': 'Bearer ' + token
            },
            body: JSON.stringify({ name, price, url })
        })
        .then(r => r.json().then(data => ({status: r.status, data})))
        .then(({status, data}) => {
            progress.style.display = 'none';
            if (status === 201) {
                document.getElementById('add-success').textContent = '¡Producto añadido!';
                document.getElementById('add-error').textContent = '';
                document.getElementById('product-name').value = '';
                document.getElementById('product-price').value = '';
            } else {
                // Mostrar todos los errores de validación si existen
                if (data.errors) {
                    const errores = Object.values(data.errors).flat().join(' ');
                    document.getElementById('add-error').textContent = errores;
                } else {
                    document.getElementById('add-error').textContent = data.error || 'Error al añadir';
                }
                document.getElementById('add-success').textContent = '';
            }
        })
        .catch(() => {
            progress.style.display = 'none';
            document.getElementById('add-error').textContent = 'Error de red';
            document.getElementById('add-success').textContent = '';
        });
    });
}

// Fusionar toda la inicialización en un solo bloque DOMContentLoaded

document.addEventListener('DOMContentLoaded', function() {
    // Mostrar header siempre (usuario o wishio)
    function updateHeaderAndSection() {
        chrome.storage.local.get(['wishio_token', 'wishio_username'], (result) => {
            const logoutBtn = document.getElementById('logout-btn');
            if (result.wishio_token && result.wishio_username) {
                setHeaderUser(result.wishio_username);
                showAdd();
                fillUrlFromTab();
                if (logoutBtn) {
                    logoutBtn.style.display = 'flex';
                }
            } else {
                setHeaderUser(null);
                showLogin();
                if (logoutBtn) logoutBtn.style.display = 'none';
            }
        });
    }
    updateHeaderAndSection();

    const usernameInput = document.getElementById('username');
    const passwordInputLogin = document.getElementById('password');
    const loginBtn = document.getElementById('login-btn');

    function triggerLogin() {
        const username = usernameInput.value;
        const password = passwordInputLogin.value;
        login(username, password);
    }

    loginBtn.onclick = triggerLogin;

    usernameInput.addEventListener('keydown', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            triggerLogin();
        }
    });
    passwordInputLogin.addEventListener('keydown', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            triggerLogin();
        }
    });

    // Tras login exitoso, actualizar header y sección
    const originalLogin = login;
    login = function(username, password) {
        originalLogin(username, password);
        // Esperar un poco para que el token se guarde y luego actualizar header
        setTimeout(updateHeaderAndSection, 400);
    };

    function triggerAddProduct() {
        getToken(token => {
            const name = document.getElementById('product-name').value;
            const price = document.getElementById('product-price').value;
            const url = document.getElementById('product-url').value;
            addProduct(token, name, price, url);
        });
    }
    document.getElementById('add-btn').onclick = triggerAddProduct;
    // Permitir añadir producto pulsando Enter en cualquier campo del formulario
    ['product-name', 'product-price', 'product-url'].forEach(id => {
        document.getElementById(id).addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                triggerAddProduct();
            }
        });
    });

    document.getElementById('logout-btn').onclick = function() {
        clearToken();
        document.getElementById('username').value = '';
        document.getElementById('password').value = '';
        chrome.storage.local.remove(['wishio_avatar']);
        setTimeout(updateHeaderAndSection, 200);
    };

    // --- Mostrar/Ocultar contraseña ---
    const passwordInput = document.getElementById('password'); // para mostrar/ocultar contraseña
    const togglePasswordBtn = document.getElementById('toggle-password');
    const eyeIcon = document.getElementById('eye-icon');
    // SVGs para ojo y ojo tachado
    function createEyeSvg() {
        const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
        svg.setAttribute('id', 'eye-icon');
        svg.setAttribute('fill', 'none');
        svg.setAttribute('viewBox', '0 0 24 24');
        svg.setAttribute('stroke-width', '1.5');
        svg.setAttribute('stroke', '#64748b');
        svg.setAttribute('style', 'width:20px;height:20px;');
        const path1 = document.createElementNS('http://www.w3.org/2000/svg', 'path');
        path1.setAttribute('stroke-linecap', 'round');
        path1.setAttribute('stroke-linejoin', 'round');
        path1.setAttribute('d', 'M2.036 12.322a1.012 1.012 0 0 1 0-.639C3.423 7.51 7.36 4.5 12 4.5c4.638 0 8.573 3.007 9.963 7.178.07.207.07.431 0 .639C20.577 16.49 16.64 19.5 12 19.5c-4.638 0-8.573-3.007-9.963-7.178Z');
        const path2 = document.createElementNS('http://www.w3.org/2000/svg', 'path');
        path2.setAttribute('stroke-linecap', 'round');
        path2.setAttribute('stroke-linejoin', 'round');
        path2.setAttribute('d', 'M15 12a3 3 0 1 1-6 0 3 3 0 0 1 6 0Z');
        svg.appendChild(path1);
        svg.appendChild(path2);
        return svg;
    }
    function createEyeSlashSvg() {
        const svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
        svg.setAttribute('id', 'eye-icon');
        svg.setAttribute('fill', 'none');
        svg.setAttribute('viewBox', '0 0 24 24');
        svg.setAttribute('stroke-width', '1.5');
        svg.setAttribute('stroke', '#64748b');
        svg.setAttribute('style', 'width:20px;height:20px;');
        const path1 = document.createElementNS('http://www.w3.org/2000/svg', 'path');
        path1.setAttribute('stroke-linecap', 'round');
        path1.setAttribute('stroke-linejoin', 'round');
        path1.setAttribute('d', 'M3.98 8.223A10.477 10.477 0 0 0 1.934 12C3.226 16.338 7.244 19.5 12 19.5c.993 0 1.953-.138 2.863-.395M6.228 6.228A10.451 10.451 0 0 1 12 4.5c4.756 0 8.773 3.162 10.065 7.498a10.522 10.522 0 0 1-4.293 5.774M6.228 6.228 3 3m3.228 3.228 3.65 3.65m7.894 7.894L21 21m-3.228-3.228-3.65-3.65m0 0a3 3 0 1 0-4.243-4.243m4.242 4.242L9.88 9.88');
        svg.appendChild(path1);
        return svg;
    }
    togglePasswordBtn.addEventListener('click', function() {
        // Eliminar SVG anterior
        while (togglePasswordBtn.firstChild) {
            togglePasswordBtn.removeChild(togglePasswordBtn.firstChild);
        }
        if (passwordInput.type === 'password') {
            passwordInput.type = 'text';
            togglePasswordBtn.appendChild(createEyeSlashSvg());
        } else {
            passwordInput.type = 'password';
            togglePasswordBtn.appendChild(createEyeSvg());
        }
    });
});
